
# language_tui.py: text mode language selection dialog
# Copyright (C) 2002-2015 Red Hat, Inc.
# Copyright (C) 2002, 2003 Brent Fox <bfox@redhat.com>
# Copyright (C) 2013-2015 Parag Nemade <pnemade@redhat.com>
##
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
##
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
##
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

import snack
import locale
import language_backend
import install_packages3
import subprocess
import gettext

locale.setlocale(locale.LC_ALL, "")
_ = lambda x: gettext.translation(
    "system-config-language", fallback=True).gettext(x) if x != "" else ""
N_ = lambda x: x
gettext.bind_textdomain_codeset("system-config-language",
                                locale.nl_langinfo(locale.CODESET))


class LanguageWindow(object):

    def __init__(self):
        self.lang_dict = {}
        self.langlist_with_cc = ['zh_CN', 'zh_TW', 'pt_BR']
        self.installed_list = []
        self.lang_backend = language_backend.LanguageBackend()
        self.list_to_install = []
        self.lang_backend = language_backend.LanguageBackend()
        self.default_lang, self.installed_langs = \
            self.lang_backend.get_installed_langs()

    @classmethod
    def lang_already_installed(cls, langid):
        imsg = _(" language support is already installed, Therefore "
                 "setting only the default system language.")
        imsg = langid + imsg
        screen = snack.SnackScreen()
        bb = snack.ButtonBar(screen, ["OK"])
        text_box = snack.TextboxReflowed(40, imsg)
        g = snack.GridFormHelp(screen, _("Language already installed"),
                               "kbdtype", 1, 4)
        g.add(text_box, 0, 0)
        g.add(bb, 0, 3, growx=1)
        rc = g.runOnce()
        button = bb.buttonPressed(rc)
        if button == "ok":
            screen.finish()

    @classmethod
    def no_pkg_support_available(cls, langid):
        imsg = _("No package support is available for language ")
        imsg = imsg + langid
        screen = snack.SnackScreen()
        bb = snack.ButtonBar(screen, ["OK"])
        text_box = snack.TextboxReflowed(40, imsg)
        g = snack.GridFormHelp(screen, _("No packages available"),
                               "kbdtype", 1, 4)
        g.add(text_box, 0, 0)
        g.add(bb, 0, 3, growx=1)
        rc = g.runOnce()
        button = bb.buttonPressed(rc)
        if button == "ok":
            screen.finish()

    @classmethod
    def install_lang_support(cls, langid):
        imsg = _(" language support is now installed. Setting"
                 " it as the default system language.")
        imsg = langid + imsg
        screen = snack.SnackScreen()
        bb = snack.ButtonBar(screen, ["OK"])
        text_box = snack.TextboxReflowed(40, imsg)
        g = snack.GridFormHelp(screen, _("Language installed"),
                               "kbdtype", 1, 4)
        g.add(text_box, 0, 0)
        g.add(bb, 0, 3, growx=1)
        rc = g.runOnce()
        button = bb.buttonPressed(rc)
        if button == "ok":
            screen.finish()

    @classmethod
    def mark_lang_is_installed(cls, langid):
        imsg = _(" language support packages are already "
                 "installed, Therefore setting only "
                 "the default system language.")
        imsg = langid + imsg
        screen = snack.SnackScreen()
        bb = snack.ButtonBar(screen, ["OK"])
        text_box = snack.TextboxReflowed(40, imsg)
        g = snack.GridFormHelp(screen, _("Language packages "
                                         "already installed"), "kbdtype", 1, 4)
        g.add(text_box, 0, 0)
        g.add(bb, 0, 3, growx=1)
        rc = g.runOnce()
        button = bb.buttonPressed(rc)
        if button == "ok":
            screen.finish()

    def lang_not_installed(self, instpkgs, langid, langcode):
        pkg_list = instpkgs.get_language_packages(langcode)

        if not pkg_list:
            self.no_pkg_support_available(langid)
        else:
            for pkgs in instpkgs.uniq_installed_pkgs:
                self.installed_list.append(pkgs)
            for name in pkg_list:
                if name not in self.installed_list:
                    self.list_to_install.append(name)
            cmd = "pkcon install "
            if self.list_to_install:
                for item in self.list_to_install:
                    cmd = cmd + " {0}".format(item)

                subprocess.check_output(cmd, shell=True)
                self.install_lang_support(langid)
                instpkgs.add_langpack_to_installed_list(langcode)
            else:
                self.mark_lang_is_installed(langid)
                instpkgs.add_langpack_to_installed_list(langcode)

    def run_ui(self, screen):
        langid = self.lang_backend.remove_encoding(self.default_lang)
        instpkgs = install_packages3.InstallLangpack()
        if langid not in self.langlist_with_cc:
            langcode = langid.split("_")[0]
        else:
            langcode = langid
        installed = instpkgs.check_if_already_installed(langcode)
        if installed is "no":
            screen.finish()
            cstr = _(" language support is not installed, do you want to "
                     "install it?")
            cstr = langid + cstr
            cscreen = snack.SnackScreen()
            bb = snack.ButtonBar(cscreen, ["OK", "Cancel"])
            text_box = snack.TextboxReflowed(40, cstr)
            # TRANSLATORS: Confirming the installation
            gc = snack.GridFormHelp(screen, _("Installing Confirm"),
                                    "kbdtype", 1, 4)
            gc.add(text_box, 0, 0)
            gc.add(bb, 0, 3, growx=1)
            rc = gc.runOnce()
            button = bb.buttonPressed(rc)
            if button == "ok":
                cscreen.finish()
                self.lang_not_installed(instpkgs, langid, langcode)
        else:
            self.lang_already_installed(langid)

    def draw_ui(self, screen):
        bb = snack.ButtonBar(screen, ["Yes", "No"])
        text_box = snack.TextboxReflowed(40,
                                         _("Select the language for the system."))
        llist = self.populate_listbox(self.default_lang)
        g = snack.GridFormHelp(
            screen, _("Language Selection"), "kbdtype", 1, 4)
        g.add(text_box, 0, 0)
        g.add(llist, 0, 1, padding=(0, 1, 0, 1))
        g.add(bb, 0, 3, growx=1)
        rc = g.runOnce()
        button = bb.buttonPressed(rc)
        if button == "no":
            return -1

        choice = llist.current()
        self.default_lang, sysfontacm, sysfont = self.lang_dict[choice]
        self.run_ui(screen)

        if self.installed_langs == None:
            self.lang_backend.write_locale_conf(self.default_lang, "",
                                                sysfont, sysfontacm)
        else:
            modules = self.installed_langs[0]
            for lang in self.installed_langs[1:]:
                modules = modules + ":" + lang

            self.lang_backend.write_locale_conf(self.default_lang, modules,
                                                sysfont, sysfontacm)

    def populate_listbox(self, deflang):
        llist = snack.Listbox(8, scroll=1, returnExit=0)
        lines = self.lang_backend.read_table()

        if deflang == None:
            llist.append('English (USA)', 'en_US.UTF-8')
            self.installed_langs = ['en_US.UTF-8:en']
            llist.setCurrent("en_US.UTF-8")
            return llist

        default = ""
        for line in lines:
            tokens = line.split()

            if self.installed_langs == None:
                lang_base = self.lang_backend.remove_encoding(tokens[0])
                name = ""
                for token in tokens[3:]:
                    name = name + " " + token

                    if self.lang_backend.remove_encoding(deflang) == lang_base:
                        default = tokens[0]

                llist.append(name, tokens[0])
                self.lang_dict[tokens[0]] = (tokens[0], tokens[1], tokens[2])

            else:
                if '.' in tokens[0]:
                    # Chop encoding off so we can compare to
                    # self.installed_langs
                    lang_base = self.lang_backend.remove_encoding(tokens[0])
                    if lang_base in self.installed_langs:
                        name = ""
                        for token in tokens[3:]:
                            name = name + " " + token

                        if self.lang_backend.remove_encoding(deflang) == \
                                lang_base:
                            default = tokens[0]

                        llist.append(name, tokens[0])
                        self.lang_dict[tokens[0]] = (tokens[0], tokens[1],
                                                     tokens[2])

        llist.setCurrent(default)
        return llist


class TUIWindow(object):

    def __init__(self):
        screen = snack.SnackScreen()
        screen.drawRootText(0, 0, "system-config-language - (C) 2004-2015 "
                            "Red Hat, Inc.")
        lw = LanguageWindow()

        finished = 0
        while not finished:
            rc = lw.draw_ui(screen)
            if rc == -1:
                screen.finish()
                finished = 1
            else:
                screen.finish()
                finished = 1
