
# language_gui.py - Contains the UI code needed for system-config-language
# Copyright (C) 2002-2015 Red Hat, Inc.
# Copyright (C) 2002, 2003, 2004, 2005, 2006, 2007 Brent Fox <bfox@redhat.com>
# Copyright (C) 2013-2018 Parag Nemade <pnemade@redhat.com>
##
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
##
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
##
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

from __future__ import absolute_import

import gi
gi.require_version('Gtk', '3.0')
from gi.repository import Gtk
import sys
import language_backend
import install_packages3

import gettext
_ = lambda x: gettext.translation(
    "system-config-language", fallback=True).gettext(x) if x != "" else ""
N_ = lambda x: x


class ChildWindow(Gtk.ApplicationWindow):

    ''' This class provides GUI for installing language support '''

    def __init__(self, app):
        ''' This is init method '''
        Gtk.Window.__init__(self, title=_("Language Selection"), application=app)

        self.msglable = Gtk.Label.new(_("Please select the default language for "
                                        "the system."))
        self.msglable.set_line_wrap(True)
        self.msglable.set_alignment(0.0, 0.5)
        self.msglable.set_padding(7, 0)

        self.listmodel = Gtk.ListStore(str, str, str, str)
        self.lang_backend = language_backend.LanguageBackend()
        self.instpkg = install_packages3.InstallLangpack()

        default_lang, self.installed_langs = \
            self.lang_backend.get_installed_langs()
        self.original_lang = default_lang
        self.lang_changed_flag = 0

        self.fill_store()

        self.view = Gtk.TreeView(model=self.listmodel)
        self.view.set_headers_visible(False)

        cell4 = Gtk.CellRendererText()
        self.langname_col = Gtk.TreeViewColumn(_("Language"), cell4, text=3)

        self.view.append_column(self.langname_col)

        self.scrolled_window = Gtk.ScrolledWindow()
        self.scrolled_window.set_border_width(10)
        self.scrolled_window.set_policy(Gtk.PolicyType.NEVER,
                                        Gtk.PolicyType.ALWAYS)
        self.scrolled_window.set_min_content_height(600)
        self.scrolled_window.set_min_content_width(600)

        self.scrolled_window.add(self.view)

        self.selection = self.view.get_selection()
        self.selection.connect("changed", self.on_changed)

        self.label = Gtk.Label()
        self.label.set_text("")

        self.box = Gtk.Box(homogeneous=False, spacing=6)
        self.box.set_orientation(orientation=Gtk.Orientation.VERTICAL)

        self.box1 = Gtk.Box(homogeneous=False, spacing=6)
        self.box1.pack_start(self.msglable, True, True, 0)

        self.box2 = Gtk.Box(homogeneous=False, spacing=6)
        self.box2.pack_start(self.scrolled_window, True, True, 0)

        self.box3 = Gtk.Box(homogeneous=True, spacing=6)
        self.button_help = Gtk.Button.new_with_mnemonic(_('_Help'))
        self.button_help.connect("clicked", self.help_btn)
        self.box3.pack_start(self.button_help, True, True, 5)
        # TRANSLATORS: GUI button to revert the system language to English
        self.button_sys_def = Gtk.Button.new_with_mnemonic(
            _('System _Defaults'))
        self.button_sys_def.connect("clicked", self.set_system_defaults)
        self.box3.pack_start(self.button_sys_def, True, True, 5)

        self.button_cancel = Gtk.Button.new_with_mnemonic(_('_Cancel'))
        self.button_cancel.connect("clicked", self.cancel_btn)
        self.box3.pack_start(self.button_cancel, True, True, 5)

        self.button_ok = Gtk.Button.new_with_mnemonic(_('_Ok'))
        self.button_ok.connect("clicked", self.ok_btn)
        self.box3.pack_start(self.button_ok, True, True, 5)

        self.box4 = Gtk.Box(homogeneous=False, spacing=6)

        self.box.pack_start(self.box1, True, True, 0)
        self.box.pack_start(self.box2, True, True, 0)
        self.box.pack_start(self.box3, True, True, 0)
        self.box.pack_start(self.box4, True, True, 0)

        self.add(self.box)
        self.set_default_lang(default_lang)
        self.lang_changed_to_default()

    def fill_store(self):
        ''' This will create the available language support list '''
        lines = self.lang_backend.read_table()

        # If /etc/sysconfig/i18n file is empty for some reason,
        # assume English is the only lang
        if self.original_lang == None:
            itr = self.listmodel.append()
            self.listmodel.set_value(itr, 0, 'en_US.UTF-8')
            self.listmodel.set_value(itr, 1, 'iso01')
            self.listmodel.set_value(itr, 2, 'lat0-sun16')
            self.listmodel.set_value(itr, 3, 'English (USA)')
            self.installed_langs = ['en_US.UTF-8:en']
            return

        for line in lines:
            tokens = line.split()
            if self.installed_langs == None:
                itr = self.listmodel.append()
                self.listmodel.set_value(itr, 0, tokens[0])
                self.listmodel.set_value(itr, 1, tokens[1])
                self.listmodel.set_value(itr, 2, tokens[2])
                name = ""
                for token in tokens[3:]:
                    name = name + " " + token
                self.listmodel.set_value(itr, 3, name)

            else:
                if '.' in tokens[0]:
                    # Chop encoding off so we can compare to
                    # self.installed_langs
                    lang_base = self.lang_backend.remove_encoding(tokens[0])

                    if lang_base in self.installed_langs:
                        itr = self.listmodel.append()
                        self.listmodel.set_value(itr, 0, tokens[0])
                        self.listmodel.set_value(itr, 1, tokens[1])
                        self.listmodel.set_value(itr, 2, tokens[2])
                        name = ""
                        for token in tokens[3:]:
                            name = name + " " + token
                        self.listmodel.set_value(itr, 3, name)

    def on_changed(self, selection):
        ''' Handler for when focus changed on the list row '''
        (model, itr) = selection.get_selected()
        if itr is not None:
            self.label.set_text("%s %s" % (model[itr][0], model[itr][3]))
        else:
            self.label.set_text("")
        return True

    def help_btn(self, button):
        ''' Method to handle Help button click '''
        sdlg = Gtk.MessageDialog(transient_for=self, modal=True,
                                 message_type=Gtk.MessageType.INFO, buttons=Gtk.ButtonsType.OK,
                                 text=(_("system-config-language is a graphical user "
                                         "interface \nthat allows the user to change the "
                                         "default language \nof the system.")))
        rc = sdlg.run()
        sdlg.destroy()

        if rc == Gtk.ResponseType.OK:
            return 1

    def langpack_already_installed(self):
        ''' Method to check if requested language langpack is
            already installed '''
        sdlg = Gtk.MessageDialog(transient_for=self, modal=True,
                                 message_type=Gtk.MessageType.INFO, buttons=Gtk.ButtonsType.OK,
                                 text=(_("Language support already installed. Therefore, "
                                         "\nsetting the default language only.")))
        rc = sdlg.run()
        sdlg.destroy()

        if rc == Gtk.ResponseType.OK:
            return 1

    def set_system_defaults(self, button):
        ''' Method to set default system language to en_US.UTF-8 '''
        rc = self.selection.get_selected()
        if rc:
            model, itr = rc
            default_lang = self.listmodel.get_value(itr, 0)

        if default_lang == "en_US.UTF-8":
            dlg = Gtk.MessageDialog(transient_for=self, modal=True,
                                    message_type=Gtk.MessageType.INFO, buttons=Gtk.ButtonsType.OK,
                                    text=(_("Already system in default language,"
                                            "\nNothing to do.")))
            rc = dlg.run()
            dlg.destroy()

            if rc == Gtk.ResponseType.OK:
                return 1

        mstr = _("Do you really want to change system language \nto default "
                 "[en_US]?")
        d = Gtk.MessageDialog(transient_for=self, modal=True,
                              message_type=Gtk.MessageType.QUESTION, buttons=Gtk.ButtonsType.OK_CANCEL,
                              text=mstr)

        d.set_default_response(Gtk.ResponseType.OK)
        rc = d.run()
        d.destroy()

        if rc == Gtk.ResponseType.OK:
            default_lang = "en_US"
            self.set_default_lang(default_lang)
        else:
            return -1

    def set_default_lang(self, default_lang):
        ''' Method to set default language '''
        itr = self.listmodel.get_iter_first()
        while itr:
            lang_base = self.lang_backend.remove_encoding(
                self.listmodel.get_value(itr, 0))
            if lang_base == default_lang:
                default_lang = self.listmodel.get_value(itr, 0)
                path = self.listmodel.get_path(itr)
                self.view.set_cursor(path, self.langname_col, False)
                break
            itr = self.listmodel.iter_next(itr)

    def lang_changed_to_default(self):
        ''' Method to check if newly selected language is the current
            default language already '''
        (model, itr) = self.selection.get_selected()
        if itr is not None:
            selected_lang = self.lang_backend.remove_encoding(
                self.listmodel.get_value(itr, 0))
            if selected_lang == self.original_lang:
                return 1
        return 0

    def langpkg_already_installed(self):
        ''' Method to check if requested language packages are
            already installed '''
        sdlg = Gtk.MessageDialog(transient_for=self, modal=True,
                                 message_type=Gtk.MessageType.INFO, buttons=Gtk.ButtonsType.OK,
                                 text=(_("Language support packages are already installed. "
                                         "Therefore, \nsetting the default language only.")))
        rc = sdlg.run()
        sdlg.destroy()

        if rc == Gtk.ResponseType.OK:
            return 1

    def no_pkg_support(self):
        """ There are no packages available to support this language """
        sdlg = Gtk.MessageDialog(transient_for=self, modal=True,
                                 message_type=Gtk.MessageType.INFO, buttons=Gtk.ButtonsType.OK,
                                 text=(_("There are no packages available yet to support this "
                                         "language. Therefore, \nsetting the default language only.")))
        rc = sdlg.run()
        sdlg.destroy()

        if rc == Gtk.ResponseType.OK:
            return 1

    def no_langpack_pkg_support(self):
        """ There is no langpack package available to support this language """
        sdlg = Gtk.MessageDialog(transient_for=self, modal=True,
                                 message_type=Gtk.MessageType.INFO, buttons=Gtk.ButtonsType.OK,
                                 text=(_("There is no langpack package available yet to support this "
                                         "language. Therefore, \nsetting the default language only.")))
        rc = sdlg.run()
        sdlg.destroy()

        if rc == Gtk.ResponseType.OK:
            return 1

    def installed_pkg_support(self, d_lang):
        """ Language packages are installed now for the requested language """
        msgd = d_lang + _(" language support is now installed.\nSetting it as "
                          "the default system language.")
        sdlg = Gtk.MessageDialog(transient_for=self, modal=True,
                                 message_type=Gtk.MessageType.INFO, buttons=Gtk.ButtonsType.OK,
                                 text=msgd)
        rc = sdlg.run()
        sdlg.destroy()

        if rc == Gtk.ResponseType.OK:
            return 1

    def ok_btn(self, button):
        ''' Method to handle Ok button click '''
        (model, itr) = self.selection.get_selected()
        if itr is not None:
            default_lang = model[itr][0]
            sysfontacm = model[itr][1]
            sysfont = model[itr][2]

        if default_lang.endswith(".UTF-8"):
            dlang = default_lang[:-6]
            installed = self.instpkg.install_packages3(dlang)
        else:
            dlang = default_lang
            installed = self.instpkg.install_packages3(dlang)
        if installed == "LangPackAlreadyInstalled":
            self.langpack_already_installed()
        elif installed == "LangPkgAlreadyInstalled":
            self.langpkg_already_installed()
        elif installed == "LangPkgNoSupport":
            self.no_pkg_support()
        elif installed == "LangSupportInstalled":
            self.installed_pkg_support(dlang)
        elif installed == "Langpack does not exists":
            self.no_langpack_pkg_support()

        if self.installed_langs == None:
            self.lang_backend.write_locale_conf(default_lang, "",
                                                sysfont, sysfontacm)
        else:
            modules = self.installed_langs[0]
            for lang in self.installed_langs[1:]:
                modules = modules + ":" + lang
            self.lang_backend.write_locale_conf(default_lang, modules,
                                                sysfont, sysfontacm)

        self.destroy()
        sys.exit()

    def cancel_btn(self, button):
        ''' Method to handle Cancel button click '''
        self.destroy()
        sys.exit()


class GUIWindow(Gtk.Application):

    ''' Class to call actual GUI methods '''

    def __init__(self):
        ''' init method '''
        Gtk.Application.__init__(self)

    def do_activate(self):
        win = ChildWindow(self)
        win.show_all()

    def do_startup(self):
        Gtk.Application.do_startup(self)
