
# languageBackend.py - The backend code needed for system-config-language
# Copyright (C) 2002-2015 Red Hat, Inc.
# Copyright (C) 2002, 2003 Brent Fox <bfox@redhat.com>
# Copyright (C) Parag Nemade <pnemade@redhat.com>
##
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
##
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
##
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

from __future__ import print_function

import string
import os

import gettext
_ = lambda x: gettext.translation(
    "system-config-language", fallback=True).gettext(x) if x != "" else ""


class LanguageBackend(object):

    def __init__(self):
        self.original_file = None
        self.path = '/etc/locale.conf'

    def get_installed_langs(self):
        if os.access(self.path, os.R_OK) == 0:
            return None, None
        else:
            fd = open(self.path, "r")
            self.original_file = fd.readlines()
            fd.close()

            default_lang = None
            langs = None

            for line in self.original_file:
                if line[:5] == "LANG=":
                    default_lang = line[5:].replace('"', '')
                    default_lang = default_lang.strip()
                if line[:9] == "SUPPORTED":
                    langs = string.replace(line[10:], '"', '')
                    langs = string.strip(langs)
                    langs = string.split(langs, ':')

            if langs:
                for lang in langs:
                    # Chop off any encoding data.  We don't really need it
                    lang_base = self.remove_encoding(lang)
                    langs[langs.index(lang)] = lang_base

            if default_lang:
                default_lang = self.remove_encoding(default_lang)
            else:
                default_lang = "en_US"

            return default_lang, langs

    @classmethod
    def remove_encoding(cls, lang):
        if '@' in lang:
            lang_base = lang.split('@')
            return lang_base[0]
        elif '.' in lang:
            lang_base = lang.split('.')
            return lang_base[0]
        else:
            return lang

    @classmethod
    def read_table(cls):
        lines = None
        fd = None
        try:
            fd = open("locale-list", "r")
        except IOError:
            try:
                fd = open("/usr/share/system-config-language/locale-list", "r")
            except IOError:
                pass
        if fd:
            lines = fd.readlines()
            fd.close()

        if not lines:
            raise RuntimeError(_("Cannot find locale-list"))
        else:
            return lines

    def write_new_path(self, default_lang, locale_path):
        fd = open(locale_path, 'w')
        if self.original_file:
            for line in self.original_file:
                if line[:5] == "LANG=":
                    fd.write('LANG="' + default_lang + '"\n')
                    # horrible hack to make simplified chinese work
                    if default_lang == "zh_CN.GB18030":
                        fd.write('LANGUAGE=  \
                                       "zh_CN.GB18030:zh_CN.GB2312:zh_CN"\n')
                elif line[:9] == 'LANGUAGE=':
                    # horrible hack to make simplified chinese work
                    if default_lang != "zh_CN.GB18030":
                        pass
                else:
                    fd.write(line)
        else:
            fd.write('LANG="' + default_lang + '"\n')

    def write_locale_conf(self, default_lang, modules, sysfont, sysfontacm):
        locale_path = '/etc/locale.conf'
        old_path = '/etc/sysconfig/i18n'
        vsconsole_file = "/etc/vconsole.conf"

        if os.access(locale_path, os.R_OK) != 0:
            self.write_new_path(default_lang, locale_path)
            fd = open(vsconsole_file, "r")
            vsconsole_data = fd.readlines()
            fd.close()
            fd = open(vsconsole_file, "w")
            for line in vsconsole_data:
                if line[:5] == "FONT=":
                    fd.write('FONT="' + sysfont + '"\n')
                else:
                    fd.write(line)
            fd.close()
        elif os.access(old_path, os.R_OK) != 0:
            fd = open(old_path, 'w')
            if self.original_file:
                for line in self.original_file:
                    if line[:5] == "LANG=":
                        fd.write('LANG="' + default_lang + '"\n')
                        # horrible hack to make simplified chinese work
                        if default_lang == "zh_CN.GB18030":
                            fd.write('LANGUAGE=  \
                                       "zh_CN.GB18030:zh_CN.GB2312:zh_CN"\n')
                    elif line[:8] == "SYSFONT=":
                        fd.write('SYSFONT="' + sysfont + '"\n')
                    elif line[:11] == 'SYSFONTACM=':
                        fd.write('SYSFONTACM="' + sysfontacm + '"\n')
                    elif line[:9] == 'LANGUAGE=':
                        # horrible hack to make simplified chinese work
                        if default_lang != "zh_CN.GB18030":
                            pass
                    else:
                        fd.write(line)
            else:
                fd.write('LANG="' + default_lang + '"\n')
                if modules:
                    fd.write('SUPPORTED="' + modules + '"\n')
                    fd.write('SYSFONT="' + sysfont + '"\n')
                if sysfontacm != "utf8":
                    fd.write('SYSFONTACM="' + sysfontacm + '"\n')
                    # horrible hack to make simplified chinese work
                if default_lang == "zh_CN.GB18030":
                    fd.write('LANGUAGE="zh_CN.GB18030:zh_CN.GB2312:zh_CN"\n')
            fd.close()
        else:
            print(
                "You should have /etc/locale.conf installed by systemd package")
