
# install_packages3.py - Language Support Installtion using PackageKit
#
# Copyright (C) 2013-2018 Red Hat, Inc.
# Authors: Parag Nemade <pnemade@redhat.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.

from __future__ import print_function

import subprocess
import os
import gettext
import dnf

_ = lambda x: gettext.translation(
    "system-config-language", fallback=True).gettext(x) if x != "" else ""
N_ = lambda x: x

class InstallLangpack(object):

    """ Class for installing packages using PackageKit """

    def __init__(self):
        """ init method """
        self.base = dnf.Base()
        self.list_to_install = []
        self.uniq_installed_pkgs = []
        self.langlist_with_cc = ['zh_CN', 'zh_TW', 'pt_BR']
        self.conffile = '/var/lib/dnf/plugins/langpacks/installed_langpacks'
        # we are not sure if conffile already exists on the system or
        # user moved or deleted it. To make sure we have conffile before
        # using any langpacks command let's try to create it first.
        self.conffile_dir = os.path.dirname(self.conffile)
        if not os.path.exists(self.conffile_dir):
            try:
                os.makedirs(self.conffile_dir)
            except (IOError, OSError) as fperror:
                print('Unable to create file : %s' % self.conffile)
                return
            try:
                with open(self.conffile, 'a'):
                    pass
            except IOError:
                print("Unable to create installed_langpacks file")

    def write_installed_langpacks(self, langplist):
        """ Write the installed langpacks from conffile """

        if not self.conffile:
            return
        confdir = os.path.dirname(self.conffile)
        if not os.path.exists(confdir):
            try:
                os.makedirs(confdir, 0o755)
            except (IOError, OSError) as e:
                return
        try:
            tmp = open(self.conffile + ".tmp", "w+")
            for line in langplist:
                tmp.write(line + "\n")
            tmp.close()
            os.rename(self.conffile + ".tmp", self.conffile)
        except (IOError, OSError) as e:
            return

    def read_installed_langpacks(self):
        """ Read the installed langpacks from conffile """

        if not self.conffile:
            return []
        ret = []
        try:
            f = open(self.conffile, "r")
            flist = f.readlines()
            f.close()
        except (IOError, OSError) as e:
            return []
        for item in flist:
            item = item.strip()
            ret.append(item)
        return ret

    def add_langpack_to_installed_list(self, lang):
        """ Add the language code to the installed languages file """
        modified = 0
        langplist = self.read_installed_langpacks()
        if lang not in langplist:
            langplist.append(lang)
            modified = 1
        if modified:
            self.write_installed_langpacks(langplist)

    def check_if_already_installed(self, check_lang):
        """ Check if requested langpack is already installed on the system """
        langpack_list = self.read_installed_langpacks()
        if check_lang not in langpack_list:
            print(("{0} language not installed").format(check_lang))
            return "no"
        else:
            print(("{0} language is already installed").format(check_lang))
            return "yes"

    @classmethod
    def get_language_packages(cls, lang):
        """ Find package list required to install the requested langpack """
        lang_pkgs = []
        cmd = "dnf repoquery --debuglevel=0 --queryformat '%{{name}}' --whatsupplements langpacks-{0}".format(lang)
        cmdoutput = subprocess.check_output(cmd, shell=True)
        for line in cmdoutput.decode('UTF-8').splitlines():
            lang_pkgs.append(line)
        print("Langpacks provides packages => {0}".format(lang_pkgs))

        return lang_pkgs

    @classmethod
    def get_matches(cls, availpkg, llist):
        ret = []
        for match in llist:
            try:
                p = availpkg.filter(provides=match)
                if p[0].name:
                    ret.append(p[0].name)
            except:
                pass
        return ret

    def install_packages3(self, lang):
        """ Find which packages are required for installation """
        pkgmatches = []
        ipkgs = []
        self.base.read_all_repos()
        repos = self.base.repos.iter_enabled()
        self.base.fill_sack()
        allpkg = self.base.sack.query()
        instpkg = allpkg.installed()
        availpkg = allpkg.available()
        availpkg = availpkg.latest()
        for pkg in instpkg:
            ipkgs.append(pkg.name)

        if lang not in self.langlist_with_cc:
            langcode = lang.split("_")[0]
        else:
            langcode = lang

        installed = self.check_if_already_installed(langcode)

        if installed is "no":
            all_pkg_list = self.get_language_packages(langcode)
            if all_pkg_list:

                # This is special case to cover package name man-pages-zh-CN
                # which should have been named as man-pages-zh_CN
                if lang.find("zh_CN") != -1:
                    all_pkg_list.append("man-pages-zh-CN")

                # Available in repo pattern matched pkgs
                all_match_pkg_list = self.get_matches(availpkg, all_pkg_list)

                # we want to make sure pkgs return only if
                # those pkgs are available to be installed
                for pkg in all_match_pkg_list:
                    if pkg not in ipkgs:
                        self.list_to_install.append(pkg)

                self.list_to_install.sort()
                print("Packages that can be installed=> {0}".format(
                    self.list_to_install))

                if not self.list_to_install:
                    print("Langpack already installed")
                    self.add_langpack_to_installed_list(langcode)
                    return "LangPkgAlreadyInstalled"
                else:
                    ret = self.install_language_packages(langcode)
                    return ret
            else:
                print("We got no packages for requested language")
                return "LangPkgNoSupport"
        else:
            return "LangPackAlreadyInstalled"

    @classmethod
    def install_language_packages(cls, langcode):
        """ Install package list for a given langcode """
        cmd = "dnf install -y langpacks-{0}".format(langcode)
        try:
            subprocess.check_output(cmd, shell=True)
            return "LangSupportInstalled"
        except subprocess.CalledProcessError as e:
            return "Langpack does not exists"

